﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace NOPWORKS.Devel.Application.WaveCutter
{
    public class WaveDataChunkCollection : WaveOperation
    {
        private List<WaveDataChunk> datas_;
        private int[][] channelDatas_;
        private MarkerDataCollection markers_;

        #region constructor
        //
        //  constructor
        //

        public WaveDataChunkCollection()
        {
            this.datas_ = new List<WaveDataChunk>();
            this.channelDatas_ = null;
            this.markers_ = new MarkerDataCollection();
            this.Clear();
        }

        #endregion  //  constructor

        #region property
        //
        //  property
        //

        public int[][] ChannelDatas
        {
            get { return this.channelDatas_; }
            set { this.channelDatas_ = value; }
        }

        public List<WaveDataChunk> Datas
        {
            get { return this.datas_; }
            set { this.datas_ = value; }
        }

        public MarkerDataCollection Markers
        {
            get { return this.markers_; }
            set { this.markers_ = value; }
        }

        #endregion  //  property

        #region method
        //
        //  method
        //

        public void Add(
            WaveDataChunk data)
        {
            this.datas_.Add(data);
        }

        public void AddMarker(
            int pos)
        {
            this.markers_.Add(new MarkerData(pos, ""));
        }

        public void AddSilenceToChannelData(
            WaveFmtChunk fmt,
            int blankTime)
        {
            float datasPerSec = (float)fmt.BytesPerSec / (float)fmt.BytesPerSample;
            float datasPerMsec = ((float)datasPerSec) / 1000f;
            int datasBlankTime = (int)datasPerMsec * blankTime;
            for (int ich = 0; ich < fmt.Channels; ich++)
            {
                int[] tmpData = new int[this.channelDatas_[ich].Length + (datasBlankTime * 2)];
                Array.Copy(
                    this.channelDatas_[ich],
                    0,
                    tmpData,
                    datasBlankTime - 1,
                    this.channelDatas_[ich].Length);
                this.channelDatas_[ich] = tmpData;
            }
        }

        public int CheckBlank(
            float volumeThreshold,
            float noiseThreshold,
            int blankTime,
            int datasPerthreshold,
            int dataLen,
            int index)
        {
            int indexTo = index;
            for (int ipos = index; ipos < dataLen; ipos++)
            {
                bool blankp = true;
                for (int ich = 0; ich < this.channelDatas_.Length; ich++)
                {
                    int vol = this.channelDatas_[ich][ipos];
                    if ((vol > volumeThreshold) || (vol < -volumeThreshold))
                    {
                        if (!this.CheckNoise(
                            volumeThreshold,
                            noiseThreshold,
                            datasPerthreshold,
                            this.channelDatas_[ich],
                            ipos))
                        {
                            blankp = false;
                            break;
                        }
                    }
                }
                if (!blankp)
                {
                    break;
                }
                indexTo = ipos;
            }
            return indexTo - index;
        }

        public bool CheckNoise(
            float volumeThreshold,
            float noiseThreshold,
            int datasPerthreshold,
            int[] datas,
            int index)
        {
            if (datas == null)
            {
                return false;
            }

            int indexFrom = index - (datasPerthreshold / 2);
            if (indexFrom < 0)
            {
                indexFrom = 0;
            }
            int indexTo = index + (datasPerthreshold / 2);
            if (indexTo >= datas.Length)
            {
                indexTo = datas.Length - 1;
            }

            int cnt = 0;

            for (int i = indexFrom; i <= indexTo; i++)
            {
                if ((datas[i] > volumeThreshold) || (datas[i] < -volumeThreshold))
                {
                    ++cnt;
                }
            }
            return (cnt < (datasPerthreshold * noiseThreshold / 100));
        }

        public void Clear()
        {
            this.datas_.Clear();
            this.channelDatas_ = new int[][] { };
            this.markers_.Clear();
        }

        public void ClearMarker()
        {
            this.markers_.Clear();
        }

        public void ClipSoundFromChannelData(
            WaveFmtChunk fmt,
            float volumeThreshold,
            int timeThreshold,
            float noiseThreshold,
            int blankTime)
        {
            int dataCount = this.GetChannelDataCount();
            if (dataCount <= 0)
            {
                return;
            }

            float datasPerSec = (float)fmt.SamplesPerSec;
            float datasPerMsec = datasPerSec / 1000f;
            int datasPerThreshold = (int)(datasPerMsec * timeThreshold);
            int datasBlankTime = (int)(datasPerMsec * blankTime);

            int minLen = -1;
            int maxLen = -1;
            for (int i = 0; i < this.channelDatas_.Length; i++)
            {
                if ((minLen == -1) || (minLen > this.channelDatas_[i].Length))
                {
                    minLen = this.channelDatas_[i].Length;
                }
                if ((maxLen == -1) || (maxLen < this.channelDatas_[i].Length))
                {
                    maxLen = this.channelDatas_[i].Length;
                }
            }

            //  head
            int headIndex = -1;
            for (int i = 0; i < minLen; i++)
            {
                for (int ich = 0; ich < this.channelDatas_.Length; ich++)
                {
                    float vol = (float)this.channelDatas_[ich][i];
                    if ((vol > volumeThreshold) || (vol < -volumeThreshold))
                    {
                        if (this.CheckNoise(
                            volumeThreshold,
                            noiseThreshold,
                            datasPerThreshold,
                            this.channelDatas_[ich],
                            i))
                        {
                            //  noise
                            for (int iNoise = i; iNoise < this.channelDatas_[ich].Length; iNoise++)
                            {
                                int tmpVol = this.channelDatas_[ich][iNoise];
                                if ((tmpVol <= volumeThreshold) && (tmpVol >= -volumeThreshold))
                                {
                                    break;
                                }
                                ++i;
                            }
                        }
                        else
                        {
                            //  not noise
                            if ((headIndex == -1) || (headIndex > i))
                            {
                                headIndex = i;
                            }
                            break;
                        }
                    }
                }
                if (headIndex >= 0)
                {
                    break;
                }
            }
            //  tail
            int tailIndex = -1;
            for (int i = minLen - 1; i >= 0; i--)
            {
                for (int ich = 0; ich < this.channelDatas_.Length; ich++)
                {
                    float vol = (float)this.channelDatas_[ich][i];
                    if ((vol > volumeThreshold) || (vol < -volumeThreshold))
                    {
                        if (this.CheckNoise(
                            volumeThreshold,
                            noiseThreshold,
                            datasPerThreshold,
                            this.channelDatas_[ich],
                            i))
                        {
                            //  noise
                            for (int iNoise = i; iNoise >= 0; iNoise--)
                            {
                                int tmpVol = this.channelDatas_[ich][iNoise];
                                if ((tmpVol <= volumeThreshold) && (tmpVol >= -volumeThreshold))
                                {
                                    break;
                                }
                                --i;
                            }
                        }
                        else
                        {
                            //  not noise
                            if (tailIndex < i)
                            {
                                tailIndex = i;
                            }
                            break;
                        }
                    }
                }
                if (tailIndex >= 0)
                {
                    break;
                }
            }

            for (int ich = 0; ich < this.channelDatas_.Length; ich++)
            {
                if (tailIndex >= 0)
                {
                    Array.Resize<int>(
                        ref this.channelDatas_[ich],
                        tailIndex + 1);
                }
                if (headIndex >= 0)
                {
                    int[] tmpData = new int[this.channelDatas_[ich].Length - headIndex];
                    Array.Copy(
                        this.channelDatas_[ich],
                        headIndex,
                        tmpData,
                        0,
                        this.channelDatas_[ich].Length - headIndex);
                    this.channelDatas_[ich] = tmpData;
                }
            }
        }

        public void BeSilentRegion(
            int indexFrom,
            int indexTo)
        {
            int channelCount = this.GetChannelDataCount() - 1;
            int ifrom = 
                ((indexFrom < 0) 
                ? 0 
                : ((indexFrom > channelCount) 
                    ? channelCount 
                    : indexFrom));
            int ito =
                ((indexTo < 0)
                ? 0
                : ((indexTo > channelCount)
                    ? channelCount
                    : indexTo));
            for (int ich = 0; ich < this.channelDatas_.Length; ich++)
            {
                for (int i = ifrom; i <= ito; i++)
                {
                    this.channelDatas_[ich][i] = 0;
                }
            }
        }

        public void ConvertToMonophonic(
            WaveFmtChunk fmt,
            int monoChannel)
        {
            int dataCount = this.GetChannelDataCount();
            int[] monoData = new int[dataCount];
            if (monoChannel == 0)
            {
                //  channel mix
                for (int i = 0; i < dataCount; i++)
                {
                    float vol = 0.0f;
                    for (int ich = 0; ich < fmt.Channels; ich++)
                    {
                        vol += (float)this.channelDatas_[ich][i] / (float)fmt.Channels;
                    }
                    monoData[i] = (int)vol;
                }
            }
            else
            {
                monoData = this.channelDatas_[monoChannel - 1];
            }
            this.channelDatas_ = new int[1][];
            this.channelDatas_[0] = monoData;
            fmt.Channels = 1;
            fmt.BytesPerSec = (uint)(fmt.SamplesPerSec * fmt.BitsPerSample / WaveData.BYTE_SIZE * fmt.Channels);
            fmt.BlockAlign = (ushort)(fmt.BitsPerSample / 8 * fmt.Channels);
        }

        public void CutRegion(
            int indexFrom,
            int indexTo)
        {
            int channelCount = this.GetChannelDataCount() - 1;
            int ifrom =
                ((indexFrom < 0)
                ? 0
                : ((indexFrom > channelCount)
                    ? channelCount
                    : indexFrom));
            int ito =
                ((indexTo < 0)
                ? 0
                : ((indexTo > channelCount)
                    ? channelCount
                    : indexTo));
            int newSize = channelCount - (ito - ifrom);
            for (int ich = 0; ich < this.channelDatas_.Length; ich++)
            {
                if (newSize <= 0)
                {
                    this.channelDatas_[ich] = new int[] { };
                }
                else if (ifrom == 0)
                {
                    int[] tmpData = new int[newSize];
                    Array.Copy(
                        this.channelDatas_[ich], 
                        ito + 1, 
                        tmpData, 
                        0, 
                        this.channelDatas_[ich].Length - ito - 2);
                    this.channelDatas_[ich] = tmpData;
                }
                else if (ito == channelCount)
                {
                    int[] tmpData = new int[newSize];
                    Array.Copy(this.channelDatas_[ich], tmpData, newSize);
                    this.channelDatas_[ich] = tmpData;
                }
                else
                {
                    int[] tmpData = new int[newSize];
                    Array.Copy(this.channelDatas_[ich], tmpData, ifrom);
                    Array.Copy(
                        this.channelDatas_[ich], 
                        ito + 1, 
                        tmpData, 
                        ifrom, 
                        this.channelDatas_[ich].Length - ito - 1);
                    this.channelDatas_[ich] = tmpData;
                }
            }
        }

        public int GetChannelDataCount()
        {
            int n = -1;
            if (this.channelDatas_ != null)
            {
                for (int i = 0; i < this.channelDatas_.Length; i++)
                {
                    if (n < 0)
                    {
                        n = this.channelDatas_[i].Length;
                    }
                    else if (n > this.channelDatas_[i].Length)
                    {
                        n = this.channelDatas_[i].Length;
                    }
                }
            }
            return n;
        }

        public void NormalizeChannelData(
            WaveFmtChunk fmt)
        {
            int dataCount = this.GetChannelDataCount();
            if (dataCount <= 0)
            {
                return;
            }

            int volRangeMax = fmt.GetMaxIntVolume();

            for (int ich = 0; ich < fmt.Channels; ich++)
            {
                int volMin = 0;
                int volMax = 0;
                for (int i = 0; i < this.channelDatas_[ich].Length; i++)
                {
                    int tmpVol = this.channelDatas_[ich][i];
                    if (tmpVol < volMin)
                    {
                        volMin = tmpVol;
                    }
                    if (tmpVol > volMax)
                    {
                        volMax = tmpVol;
                    }
                }
                int vol =
                    ((Math.Abs(volMin) > Math.Abs(volMax))
                    ? Math.Abs(volMin)
                    : Math.Abs(volMax));
                float rate = (float)volRangeMax / (float)vol;
                for (int i = 0; i < this.channelDatas_[ich].Length; i++)
                {
                    this.channelDatas_[ich][i] = (int)(this.channelDatas_[ich][i] * rate);
                    if (this.channelDatas_[ich][i] > volRangeMax)
                    {
                        this.channelDatas_[ich][i] = volRangeMax;
                    }
                }
            }
        }

        /// <summary>
        /// Clearing the data of channel data array.
        /// </summary>
        /// <param name="fmt">fmt chunk data</param>
        /// <param name="volumeThreshold">The volume that is smaller than volumethreshold is deleted.</param>
        /// <param name="timeThreshold">Data that continuance time is shorter than threshold is deleted.(milli sec)</param>
        public void ReduceChannelDataNoise(
            WaveFmtChunk fmt,
            float volumeThreshold,
            int timeThreshold,
            float noiseThreshold)
        {
            int dataCount = this.GetChannelDataCount();
            if (dataCount <= 0)
            {
                return;
            }

            float datasPerSec = fmt.BytesPerSec / fmt.BytesPerSample;
            float datasPerMsec = ((float)datasPerSec) / 1000f;
            int datasPerThreshold = (int)datasPerMsec * timeThreshold;
            for (int ich = 0; ich < this.channelDatas_.Length; ich++)
            {
                int pos = 0;
                while (pos < this.channelDatas_[ich].Length)
                {
                    int vol = this.channelDatas_[ich][pos];
                    if ((vol <= volumeThreshold) && (vol >= -volumeThreshold))
                    {
                        this.channelDatas_[ich][pos++] = 0;
                    }
                    else if (this.CheckNoise(
                        volumeThreshold,
                        noiseThreshold,
                        datasPerThreshold,
                        this.channelDatas_[ich],
                        pos))
                    {
                        //  noise
                        int cnt = 0;
                        for (int i = pos; i < pos + datasPerThreshold; i++)
                        {
                            if (i >= this.channelDatas_[ich].Length)
                            {
                                break;
                            }
                            else if (
                                (this.channelDatas_[ich][i] > volumeThreshold)
                                || (this.channelDatas_[ich][i] < -volumeThreshold))
                            {
                                this.channelDatas_[ich][i] = 0;
                            }
                            else
                            {
                                break;
                            }
                            ++cnt;
                        }
                        pos += cnt;
                    }
                    else
                    {
                        //  valid sound
                        do
                        {
                            ++pos;
                            if (pos >= this.channelDatas_[ich].Length)
                            {
                                break;
                            }
                            vol = this.channelDatas_[ich][pos];
                        } while ((vol > volumeThreshold) || (vol < -volumeThreshold));
                    }
                }
            }
        }

        public void ReduceClipNoise(
            WaveFmtChunk fmt,
            float volumeThresholdValue,
            int timeThreshold,
            float noiseThreshold)
        {
            int datasPerThreshold = (int)(fmt.SamplesPerSec * timeThreshold / 1000f);

            for (int ich = 0; ich < this.channelDatas_.Length; ich++)
            {
                float sum_p = 0f;
                float sum_m = 0f;
                float average_p = 0f;
                float average_m = 0f;
                float averageCnt_p = 0;
                float averageCnt_m = 0;
                for (int i = 0; i < this.channelDatas_[ich].Length; i++)
                {
                    int last = (i + datasPerThreshold <= this.channelDatas_[ich].Length) ? i + datasPerThreshold : this.channelDatas_[ich].Length;
                    if (i == 0)
                    {
                        for (int j = i; j < last; j++)
                        {
                            if (this.channelDatas_[ich][j] >= 0)
                            {
                                sum_p += this.channelDatas_[ich][j];
                                ++averageCnt_p;
                            }
                            else
                            {
                                sum_m += this.channelDatas_[ich][j];
                                ++averageCnt_m;
                            }
                        }
                    }
                    else
                    {
                        if (this.channelDatas_[ich][last - 1] >= 0)
                        {
                            sum_p += this.channelDatas_[ich][last - 1];
                            ++averageCnt_p;
                        }
                        else
                        {
                            sum_m += this.channelDatas_[ich][last - 1];
                            ++averageCnt_m;
                        }
                    }
                    average_p = sum_p / averageCnt_p;
                    average_m = sum_m / averageCnt_m;

                    int vol = this.channelDatas_[ich][i];
                    float volumeRate = 0f;
                    if (vol >= 0)
                    {
                        volumeRate = vol / average_p;
                    }
                    else
                    {
                        volumeRate = -vol / average_m;
                    }

                    if ((vol > volumeThresholdValue) || (vol < -volumeThresholdValue))
                    {
                        float volumeThreshold = 0.3f;
                        int indexFrom = (i - (datasPerThreshold / 2) < 0) ? 0 : (i - (datasPerThreshold / 2));
                        int indexTo = (i + (datasPerThreshold / 2) > this.channelDatas_[ich].Length - 1) ? this.channelDatas_[ich].Length - 1 : (i + (datasPerThreshold / 2));
                        int indexHead = -1;
                        int indexTail = -1;
                        int n = 0;
                        for (int j = i; j >= indexFrom; j--)
                        {
                            float volumeRate_j = 0f;
                            if (this.channelDatas_[ich][j] >= 0)
                            {
                                volumeRate_j = this.channelDatas_[ich][j] / average_p;
                            }
                            else
                            {
                                volumeRate_j = this.channelDatas_[ich][j] / average_m;
                            }

                            if ((volumeRate_j < volumeThreshold) && (volumeRate_j > -volumeThreshold))
                            {
                                break;
                            }
                            ++n;
                            indexHead = j;
                        }
                        for (int j = i + 1; j <= indexTo; j++)
                        {
                            float volumeRate_j = 0f;
                            if (this.channelDatas_[ich][j] >= 0)
                            {
                                volumeRate_j = this.channelDatas_[ich][j] / average_p;
                            }
                            else
                            {
                                volumeRate_j = this.channelDatas_[ich][j] / average_m;
                            }

                            if ((volumeRate_j < volumeThreshold) && (volumeRate_j > -volumeThreshold))
                            {
                                break;
                            }
                            ++n;
                            indexTail = j;
                        }

                        if (indexHead >= 0)
                        {
                            if (n / datasPerThreshold * 100 <= noiseThreshold)
                            {
                                for (int j = indexHead; j <= indexTail; j++)
                                {
                                    if (this.channelDatas_[ich][j] >= 0)
                                    {
                                        this.channelDatas_[ich][j] = (int)average_p;
                                    }
                                    else
                                    {
                                        this.channelDatas_[ich][j] = (int)average_m;
                                    }
                                }
                            }
                        }
                    }

                    if (this.channelDatas_[ich][i] >= 0)
                    {
                        sum_p -= this.channelDatas_[ich][i];
                        --averageCnt_p;
                    }
                    else
                    {
                        sum_m -= this.channelDatas_[ich][i];
                        --averageCnt_m;
                    }
                }
            }
        }

        public void RemoveAt(
            int index)
        {
            try
            {
                this.datas_.RemoveAt(index);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunkCollection.RemoveAt)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public int SetChannelData(
            WaveFmtChunk fmt)
        {
            try
            {
                this.channelDatas_ = new int[fmt.Channels][];

                int bytesPerSample = fmt.BytesPerSample;
                int channelDataCount = -1;
                int dstpos = 0;
                foreach (WaveDataChunk idata in this.datas_)
                {
                    if (!idata.DataIDString.Equals(WaveData.DATA_CHUNK_ID))
                    {
                        continue;
                    }

                    int dataSize = idata.ByteDatas.Length / fmt.BytesPerSample / (int)fmt.Channels;
                    for (int i = 0; i < fmt.Channels; i++)
                    {
                        if (this.channelDatas_[i] == null)
                        {
                            this.channelDatas_[i] = new int[dataSize];
                        }
                        else
                        {
                            Array.Resize<int>(
                                ref this.channelDatas_[i],
                                this.channelDatas_[i].Length + dataSize);
                        }
                    }

                    int pos = 0;
                    int channel = 0;
                    while (pos + bytesPerSample < idata.ByteDatas.Length)
                    {
                        int n = WaveData.ConvertBytesToInt(
                            idata.ByteDatas,
                            pos,
                            pos + bytesPerSample - 1);

                        this.channelDatas_[channel][dstpos] = n;

                        if (channel == (fmt.Channels - 1))
                        {
                            ++dstpos;
                        }
                        channel = (++channel) % fmt.Channels;
                        pos += bytesPerSample;
                    }

                    channelDataCount += dataSize;
                }
                return channelDataCount;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunkCollection.SetChannelData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool SetChannelDataToChunkData(
            WaveFmtChunk fmt)
        {
            try
            {
                int indexFrom = 0;
                int indexTo = this.GetChannelDataCount();
                return this.SetPartChannelDataToChunkData(fmt, indexFrom, indexTo);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunkCollection.SetChannelDataToChunkData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool SetChannelDataToChunkData(
            WaveFmtChunk fmt,
            int indexFrom,
            int indexTo)
        {
            try
            {
                return this.SetPartChannelDataToChunkData(fmt, indexFrom, indexTo);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunkCollection.SetChannelDataToChunkData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool SetPartChannelDataToChunkData(
            WaveFmtChunk fmt,
            int indexFrom,
            int indexTo)
        {
            try
            {
                if (this.channelDatas_ == null)
                {
                    return false;
                }
                else if (this.channelDatas_.Length < fmt.Channels - 1)
                {
                    return false;
                }
                else if (indexFrom > indexTo)
                {
                    return false;
                }

                int channelDataCounts = indexTo - indexFrom + 1;
                uint byteDataCounts = (uint)(channelDataCounts * fmt.BytesPerSample * fmt.Channels);
                int tailCount = (int)byteDataCounts;
                if (fmt.BytesPerSample > 2)
                {
                    int n = 0;
                    Math.DivRem((int)byteDataCounts, 2, out n);
                    while (n != 0)
                    {
                        byteDataCounts += (uint)(fmt.BytesPerSample * fmt.Channels);
                        Math.DivRem((int)byteDataCounts, 2, out n);
                    } 
                }

                WaveDataChunk dataChunk = null;
                foreach (WaveDataChunk idata in this.datas_)
                {
                    if (idata.DataIDString.Equals(WaveData.DATA_CHUNK_ID))
                    {
                        dataChunk = idata;
                        break;
                    }
                }
                if (dataChunk == null)
                {
                    dataChunk = new WaveDataChunk();
                    byte[] unicodeDataID = Encoding.Unicode.GetBytes(WaveData.DATA_CHUNK_ID);
                    dataChunk.DataID = Encoding.Convert(
                        Encoding.Unicode,
                        Encoding.ASCII,
                        unicodeDataID);
                    this.datas_.Add(dataChunk);
                }
                dataChunk.ByteDatas = new byte[byteDataCounts];
                dataChunk.DataSize = byteDataCounts;

                //  clearing added array data
                for (int i = tailCount - 1; i < byteDataCounts; i++)
                {
                    //  volume level of zero is '0'.
                    //  because, when the data size is larger than 16 bits, data is added to the array. 
                    dataChunk.ByteDatas[i] = 0;
                }

                int pos = 0;
                //for (int idata = 0; idata < channelDataCounts; idata++)
                for (int idata = indexFrom; idata < indexTo; idata++)
                {
                    for (int ich = 0; ich < fmt.Channels; ich++)
                    {
                        int n = 0;
                        if (this.channelDatas_[ich].Length > idata)
                        {
                            n = this.channelDatas_[ich][idata];
                        }
                        for (int ibyte = 0; ibyte < fmt.BytesPerSample; ibyte++)
                        {
                            dataChunk.ByteDatas[pos++] = (byte)(0xff & n);
                            n >>= WaveData.BYTE_SIZE;
                        }
                    }
                }

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunkCollection.SetPartChannelDataToChunkData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public int SetSplitMarker(
            WaveFmtChunk fmt,
            float volumeThreshold,
            int timeThreshold,
            float noiseThreshold,
            int blankTime)
        {
            this.markers_.Clear();

            float datasPerSec = (float)fmt.SamplesPerSec;
            int datasPerThreshold = (int)(datasPerSec * timeThreshold / 1000);
            int datasPerBlankTime = (int)(datasPerSec * blankTime / 1000);

            int minLen = this.GetMinChannelDataIndex(this.channelDatas_);

            int markerIndex = 0;
            for (int ipos = 0; ipos < minLen; ipos++)
            {
                int blankLen = this.CheckBlank(
                    volumeThreshold,
                    noiseThreshold,
                    blankTime,
                    datasPerThreshold,
                    minLen,
                    ipos);
                if (blankLen >= datasPerBlankTime)
                {
                    if ((ipos > 0) && (ipos + blankLen + 1 < minLen))
                    {
                        this.markers_.Add(ipos + (blankLen / 2), MarkerData.PREFIX + (++markerIndex).ToString());
                    }
                }
                ipos += blankLen;
            }
            return markerIndex;
        }

        public bool WriteByBinaryWriter(
            BinaryWriter writer)
        {
            try
            {
                foreach (WaveDataChunk idata in this.datas_)
                {
                    if (!idata.WriteByBinaryWriter(writer))
                    {
                        throw new Exception(
                            "(WaveDataChunkCollection.WriteByBinaryWriter(writer)\n"
                            + "[datas_(" + idata.DataIDString + ").WriteByBinaryWriter(writer)]");
                    }
                }

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunkCollection.WriteByBinaryWriter(writer)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        #endregion  //  method
    }
}
