﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;

namespace NOPWORKS.Devel.Application.WaveCutter
{
    public class WaveData : WaveOperation
    {
        public const string RIFF_ID = "RIFF";
        public const string WAVE_ID = "WAVE";
        public const string FMT_CHUNK_ID = "fmt ";
        public const string DATA_CHUNK_ID = "data";
        public const int BYTE_SIZE = 8;                         //  byte size(bit)
        public const int CHUNK_ID_DATASIZE = 4;                 //  chunk ID size(byte)
        public const int CHUNK_DATASIZE_DATASIZE = 4;           //  size of chunk data(byte)
        public const uint UINT32_MASK = 0xFFFFFFFF;             //  32bit mask
        public const int BYTEDATA_BUFFERSIZE = 1024;            //  size of read/write buffer
        //  wave data
        private byte[] fileFormatID_;                           //  'R', 'I', 'F', 'F'
        private uint fileSize_;                                 //  file size - 8(fileFormatID_ + fileSize_ = 8)
        private byte[] dataFormatID_;                           //  'W', 'A', 'V', 'E'
        private WaveFmtChunk fmtChunk_;                         //  fmt chunk
        private WaveDataChunkCollection dataChunks_;            //  data chunk
        //  private variables
        private Color graphBackColor_;
        private float volumeThreshold_;                         //  (%)
        private int timeThreshold_;                             //  (millisecond)
        private int blankTime_;                                 //  (millisecond)
        private float noiseThreshold_;                          //  (%)
        private int channel_;                                   //  (1:front left, 2: front right, 3:front center, 4:low frequency, 5:back left, 6:back right)

        #region constructor
        //
        //  constructor
        //

        public WaveData()
        {
            this.fmtChunk_ = new WaveFmtChunk();
            this.dataChunks_ = new WaveDataChunkCollection();
            this.Clear();
        }

        #endregion  //  constructor

        #region property
        //
        //  property
        //

        public int BlankTime
        {
            get { return this.blankTime_; }
            set { this.blankTime_ = value; }
        }

        public int Channel
        {
            get { return this.channel_; }
            set { this.channel_ = value; }
        }

        public WaveDataChunkCollection DataChunks
        {
            get { return this.dataChunks_; }
            set { this.dataChunks_ = value; }
        }

        public byte[] DataFormatID
        {
            get { return this.dataFormatID_; }
            set { this.dataFormatID_ = value; }
        }

        public string DataFormatIDString
        {
            get { return Encoding.UTF8.GetString(this.dataFormatID_); }
        }

        public byte[] FileFormatID
        {
            get { return this.fileFormatID_; }
            set { this.fileFormatID_ = value; }
        }

        public string FileFormatIDString
        {
            get { return Encoding.UTF8.GetString(this.fileFormatID_); }
        }

        public uint FileSize
        {
            get { return this.fileSize_; }
            set { this.fileSize_ = value; }
        }

        public WaveFmtChunk FmtChunk
        {
            get { return this.fmtChunk_; }
            set { this.fmtChunk_ = value; }
        }

        public Color GraphBackColor
        {
            get { return this.graphBackColor_; }
            set { this.graphBackColor_ = value; }
        }

        public float NoiseThreshold
        {
            get { return this.noiseThreshold_; }
            set { this.noiseThreshold_ = value; }
        }

        public int TimeThreshold
        {
            get { return this.timeThreshold_; }
            set { this.timeThreshold_ = value; }
        }

        public float VolumeThreshold
        {
            get { return this.volumeThreshold_; }
            set { this.volumeThreshold_ = value; }
        }

        #endregion  //  property

        #region method
        //
        //  method
        //

        public void AddSilence()
        {
            this.dataChunks_.AddSilenceToChannelData(this.fmtChunk_, this.blankTime_);
        }

        public void BeSilentRegion(
            int width,
            int fromX,
            int toX)
        {
            try
            {
                int indexFrom = (int)(this.CalcChannelDataStep((float)width) * (fromX));
                int indexTo = (int)(this.CalcChannelDataStep((float)width) * toX);
                int channelDataCount = this.dataChunks_.GetChannelDataCount();
                if (indexFrom < 0)
                {
                    indexFrom = 0;
                }
                else if (indexFrom >= channelDataCount)
                {
                    indexFrom = channelDataCount;
                }
                if (indexTo < 0)
                {
                    indexTo = 0;
                }
                else if (indexTo >= channelDataCount)
                {
                    indexTo = channelDataCount;
                }

                this.dataChunks_.BeSilentRegion(indexFrom, indexTo);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.EraseRegion(g, fromX, toX)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public float CalcChannelDataStep(
            float width)
        {
            if (this.dataChunks_.Datas.Count == 0)
            {
                return -1.0F;
            }

            int channelDataCount = this.dataChunks_.GetChannelDataCount();
            if (channelDataCount <= 0)
            {
                return -1.0F;
            }

            float stepData = (float)(channelDataCount) / (float)width;
            if (stepData < 1)
            {
                stepData = 1.0F;
            }

            return stepData;
        }

        public void Clear()
        {
            this.fileFormatID_ = new byte[4];
            this.fileSize_ = 0;
            this.dataFormatID_ = new byte[4];
            this.fmtChunk_.Clear();
            this.dataChunks_.Clear();
            this.graphBackColor_ = Color.FromArgb(0xFF, 0xCC, 0xDD, 0xEE);
            this.volumeThreshold_ = 1.0F;
            this.timeThreshold_ = 50;
            this.blankTime_ = 500;
            this.noiseThreshold_ = 100F;
            this.channel_ = 0;
        }

        public void ClearMarker()
        {
            this.dataChunks_.ClearMarker();
        }

        public void ClipSound()
        {
            this.dataChunks_.ClipSoundFromChannelData(
                this.fmtChunk_,
                this.GetVolumeThresholdValue(),
                this.timeThreshold_,
                this.noiseThreshold_,
                this.blankTime_);
        }

        public static int ConvertBytesToInt(
            byte[] bytes)
        {
            uint n = 0;
            if ((bytes[bytes.Length - 1] & 0x80) > 0)
            {
                n = UINT32_MASK;
            }
            for (int i = bytes.Length - 1; i >= 0; i--)
            {
                n <<= BYTE_SIZE;
                n += bytes[i];
            }
            return (int)n;
        }

        /// <summary>
        /// Converting binary data to Int32.
        /// </summary>
        /// <remarks>
        /// The binary data is a little endian.
        /// </remarks>
        /// <param name="bytes">array of binary data</param>
        /// <param name="indexFrom">conversion beginning position</param>
        /// <param name="indexTo">conversion ending position</param>
        /// <returns>converted value</returns>
        public static int ConvertBytesToInt(
            byte[] bytes,
            int indexFrom,
            int indexTo)
        {
            uint n = 0;
            if ((bytes[indexTo] & 0x80) > 0)
            {
                n = UINT32_MASK;
            }
            for (int i = indexTo; i >= indexFrom; i--)
            {
                n <<= BYTE_SIZE;
                n += bytes[i];
            }
            return (int)n;
        }

        /// <summary>
        /// Converting binary data to UInt32.
        /// </summary>
        /// <remarks>
        /// The binary data is a little endian.
        /// </remarks>
        /// <param name="bytes">array of binary data</param>
        /// <param name="indexFrom">conversion beginning position</param>
        /// <param name="indexTo">conversion ending position</param>
        /// <returns>converted value</returns>
        public static uint ConvertBytesToUInt(
            byte[] bytes,
            int indexFrom,
            int indexTo)
        {
            uint n = 0;
            for (int i = indexTo; i >= indexFrom; i--)
            {
                n <<= BYTE_SIZE;
                n += bytes[i];
            }
            return n;
        }

        public void ConvertToMonophonic()
        {
            try
            {
                this.dataChunks_.ConvertToMonophonic(this.fmtChunk_, this.channel_);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.ConvertToMonaural)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public void CutRegion(
            int width,
            int fromX,
            int toX)
        {
            try
            {
                int channelDataCount = this.dataChunks_.GetChannelDataCount();
                int[] fromIndexes = this.GetChannelDataIndexFromX(channelDataCount, width, fromX);
                int[] toIndexes = this.GetChannelDataIndexFromX(channelDataCount, width, toX);

                if (fromIndexes.Length <= 0)
                {
                    return;
                }
                else if (toIndexes.Length <= 0)
                {
                    return;
                }

                int minIndex = channelDataCount + 1;
                int maxIndex = -1;

                if (minIndex > fromIndexes[0])
                {
                    minIndex = fromIndexes[0];
                }
                if (maxIndex < fromIndexes[0])
                {
                    maxIndex = fromIndexes[0];
                }

                if (minIndex > toIndexes[0])
                {
                    minIndex = toIndexes[0];
                }
                if (maxIndex < toIndexes[0])
                {
                    maxIndex = toIndexes[0];
                }

                if ((maxIndex < 0) || (minIndex > channelDataCount))
                {
                    return;
                }

                this.dataChunks_.CutRegion(minIndex, maxIndex);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.EraseRegion(g, fromX, toX)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public void DrawWaveform(
            Graphics g,
            Rectangle selectedRect)
        {
            try
            {
                g.Clear(this.graphBackColor_);

                if (this.dataChunks_.Datas.Count == 0)
                {
                    return;
                }

                int channelDataCount = this.dataChunks_.GetChannelDataCount();
                if (channelDataCount == 0)
                {
                    return;
                }

                float h = g.VisibleClipBounds.Height / (float)fmtChunk_.Channels;
                float offsetY = h / 2;
                float w = (float)g.VisibleClipBounds.Width;
                float ratioX = (float)channelDataCount / w;
                float ratioY = offsetY / (float)fmtChunk_.GetMaxIntVolume();

                float[] x1 = new float[fmtChunk_.Channels];
                float[] x2 = new float[fmtChunk_.Channels];
                float[] y1 = new float[fmtChunk_.Channels];
                float[] y2 = new float[fmtChunk_.Channels];
                float[] offsetYCh = new float[fmtChunk_.Channels];
                for (int i = 0; i < fmtChunk_.Channels; i++)
                {
                    x1[i] = x2[i] = 0.0F;
                    y1[i] = y2[i] = 0.0F;
                    offsetYCh[i] = h * i + offsetY;
                }
                int pos = 0;
                int tmpPos = 0;
                for (float x = 0; x <= w; x += 1)
                {
                    pos = ((x < w) ? (int)(x * ratioX) : channelDataCount - 1);
                    if (pos >= channelDataCount)
                    {
                        pos = channelDataCount - 1;
                    }
                    if ((pos != tmpPos)
                        || (x == 0.0F)
                        || (x == w))
                    {
                        for (int ich = 0; ich < this.fmtChunk_.Channels; ich++)
                        {
                            int minVol = 0;
                            int maxVol = 0;
                            if (ratioX > 1.0F)
                            {
                                minVol = this.dataChunks_.ChannelDatas[ich][tmpPos];
                                maxVol = this.dataChunks_.ChannelDatas[ich][tmpPos];
                            }
                            else
                            {
                                minVol = this.dataChunks_.ChannelDatas[ich][pos];
                                maxVol = this.dataChunks_.ChannelDatas[ich][pos];
                            }
                            for (int i = tmpPos + 1; i <= pos; i++)
                            {
                                if (minVol > this.dataChunks_.ChannelDatas[ich][i])
                                {
                                    minVol = this.dataChunks_.ChannelDatas[ich][i];
                                }
                                if (maxVol < this.dataChunks_.ChannelDatas[ich][i])
                                {
                                    maxVol = this.dataChunks_.ChannelDatas[ich][i];
                                }
                            }

                            float minY = offsetYCh[ich] - (minVol * ratioY);
                            float maxY = offsetYCh[ich] - (maxVol * ratioY);

                            x2[ich] = x;
                            y2[ich] = ((Math.Abs(minVol) < Math.Abs(maxVol)) ? maxY : minY);
                            if (x == 0.0F)
                            {
                                x1[ich] = x2[ich];
                                y1[ich] = y2[ich];
                            }

                            //  normal pen
                            Pen pen = new Pen(Color.Black);
                            if ((x2[ich] >= selectedRect.X)
                                && (x2[ich] <= selectedRect.X + selectedRect.Width))
                            {
                                //  selected pen
                                pen = new Pen(Color.Blue);
                            }

                            //  volume vertical line
                            g.DrawLine(pen, x2[ich], minY, x2[ich], maxY);
                            //  volume line graph
                            g.DrawLine(pen, x1[ich], y1[ich], x2[ich], y2[ich]);

                            x1[ich] = x2[ich];
                            y1[ich] = y2[ich];
                        }

                        //  marker
                        if (this.dataChunks_.Markers.Count > 0)
                        {
                            MarkerData marker = this.dataChunks_.Markers.GetMarkerByRange(tmpPos, pos);
                            if (marker != null)
                            {
                                marker.Rect = new Rectangle((int)(x - 8), 0, 16, 16);
                                Color markerColor = marker.Selected ? Color.FromArgb(0xff, 0xff, 0xff, 0x00) : Color.FromArgb(0xff, 0x00, 0xcc, 0x00);
                                g.DrawLine(new Pen(markerColor), x, 0, x, g.ClipBounds.Height);
                                g.FillRectangle(new SolidBrush(markerColor), marker.Rect);
                            }
                        }
                    }
                    tmpPos = pos;
                }

                for (int i = 0; i < fmtChunk_.Channels; i++)
                {
                    float y = 0.0F;

                    //  0 level
                    y = h * i + offsetY;
                    g.DrawLine(new Pen(Color.FromArgb(128, 0x00, 0xff, 0x00)), 0, y, g.VisibleClipBounds.Width, y);

                    //  volume threshold
                    y = offsetY * this.volumeThreshold_ / 100 + h * i + offsetY;
                    g.DrawLine(new Pen(Color.FromArgb(128, 0xff, 0, 0)), 0, y, g.VisibleClipBounds.Width, y);
                    y = -offsetY * this.volumeThreshold_ / 100 + h * i + offsetY;
                    g.DrawLine(new Pen(Color.FromArgb(128, 0xff, 0, 0)), 0, y, g.VisibleClipBounds.Width, y);
                }

                //  selectedRect
                g.FillRectangle(
                    new SolidBrush(Color.FromArgb(128, 0xff, 0xff, 0xff)),
                    selectedRect);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.DrawWave)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public uint GetSize()
        {
                uint nsize =
                (uint)(
                sizeof(byte) * this.fileFormatID_.Length            //  fileFormatID_
                + sizeof(uint)                                      //  fileSize_
                + sizeof(byte) * this.dataFormatID_.Length          //  dataFormatID_
                + this.fmtChunk_.GetSize()                          //  fmtChunk_
                );
            foreach (WaveDataChunk idata in this.dataChunks_.Datas) //  dataChunks_
            {
                nsize += idata.GetSize();
            }
            return nsize;
        }

        public int GetVolumeThresholdValue()
        {
            return (int)(this.fmtChunk_.GetMaxIntVolume() * this.volumeThreshold_ / 100);
        }

        public void Normailze()
        {
            this.dataChunks_.NormalizeChannelData(this.fmtChunk_);
        }

        /// <summary>
        /// Reading wave data from file.
        /// </summary>
        /// <param name="fileName">Wave data file path.</param>
        /// <returns>True:read success.</returns>
        public bool ReadFromFile(
            string fileName)
        {
            BinaryReader reader = null;
            try
            {
                this.Clear();

                reader = new BinaryReader(File.OpenRead(fileName));
                if (reader.BaseStream.Length < RIFF_ID.Length + sizeof(uint) + WAVE_ID.Length)
                {
                    throw new Exception("Too short file length.");
                }

                this.fileFormatID_ = reader.ReadBytes(this.fileFormatID_.Length);
                if (!this.FileFormatIDString.Equals(WaveData.RIFF_ID))
                {
                    throw new Exception("Unknown file format.(" + this.FileFormatIDString + ")");
                }

                this.fileSize_ = reader.ReadUInt32();

                this.dataFormatID_ = reader.ReadBytes(this.dataFormatID_.Length);
                if (!this.DataFormatIDString.Equals(WaveData.WAVE_ID))
                {
                    throw new Exception("Unknown data format.(" + this.DataFormatIDString + ")");
                }

                byte[] tmpChunkIDData = null;
                string tmpChunkID = "";
                while (reader.BaseStream.Position < reader.BaseStream.Length)
                {
                    tmpChunkIDData = reader.ReadBytes(CHUNK_ID_DATASIZE);
                    tmpChunkID = Encoding.UTF8.GetString(tmpChunkIDData);
                    if (tmpChunkID == FMT_CHUNK_ID)
                    {
                        reader.BaseStream.Position -= CHUNK_ID_DATASIZE;
                        this.fmtChunk_.ReadFromBinaryReader(reader);
                    }
                    else if (tmpChunkID == DATA_CHUNK_ID)
                    {
                        reader.BaseStream.Position -= CHUNK_ID_DATASIZE;
                        WaveDataChunk tmpDataChunk = new WaveDataChunk();
                        tmpDataChunk.ReadFromBinaryReader(reader);
                        this.dataChunks_.Add(tmpDataChunk);
                    }
                    else
                    {
                        reader.BaseStream.Position -= CHUNK_ID_DATASIZE;
                        WaveDataChunk tmpDataChunk = new WaveDataChunk();
                        tmpDataChunk.ReadFromBinaryReader(reader);
                        this.dataChunks_.Add(tmpDataChunk);
                    }
                }
                SetChannelData();
                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.ReadFromFile)\n"
                    + "[" + exp.Source + "] "
                    + exp.Message + "\n",
                    exp);
            }
            finally
            {
                if (reader != null)
                {
                    reader.Close();
                }
            }
        }

        public void ReduceNoise()
        {
            this.dataChunks_.ReduceClipNoise(
                this.fmtChunk_,
                this.GetVolumeThresholdValue(),
                this.timeThreshold_,
                this.noiseThreshold_);
        }

        public void RemoveSelectedMarker()
        {
            this.dataChunks_.Markers.RemoveSelected();
        }

        public int SetChannelData()
        {
            try
            {
                return this.dataChunks_.SetChannelData(this.fmtChunk_);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.SetChannelData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        /// <summary>
        /// setting wave data for write.
        /// </summary>
        /// <returns>True:success, False:something bad</returns>
        public bool SetWaveData()
        {
            try
            {
                if (!this.dataChunks_.SetChannelDataToChunkData(this.fmtChunk_))
                {
                    return false;
                }
                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.SetWaveData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public void SplitSound()
        {
            this.dataChunks_.SetSplitMarker(
                this.fmtChunk_,
                this.GetVolumeThresholdValue(),
                this.timeThreshold_,
                this.noiseThreshold_,
                this.blankTime_);
        }

        public void ToggleMarkerSelected(
            int x,
            int y)
        {
            this.dataChunks_.Markers.ToggleMarkerSelected(x, y);
        }

        public bool WriteData(
            string fileName)
        {
            try
            {
                int channelDataCount = this.dataChunks_.GetChannelDataCount();
                bool p = true;
                if (this.dataChunks_.Markers.Count > 0)
                {
                    string baseName = fileName.Substring(0, fileName.LastIndexOf('.'));
                    string ext = fileName.Substring(fileName.LastIndexOf('.'));
                    int tmpIndex = 0;
                    int fileIndex = 1;
                    foreach (MarkerData imarker in this.dataChunks_.Markers.Markers)
                    {
                        if (imarker.Index > tmpIndex)
                        {
                            p |= this.WritePartData(baseName + "-" + fileIndex.ToString("D4") + ext, tmpIndex, imarker.Index);
                            ++fileIndex;
                        }
                        tmpIndex = imarker.Index;
                    }
                    if (tmpIndex > 0)
                    {
                        if (tmpIndex < channelDataCount)
                        {
                            p |= this.WritePartData(baseName + "-" + fileIndex.ToString("D4") + ext, tmpIndex, channelDataCount);
                            ++fileIndex;
                        }
                    }
                }
                else
                {
                    p = this.WritePartData(fileName, 0, channelDataCount);
                }
                return p;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.WriteData(fileName))\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool WriteData(
            string fileName,
            int indexFrom,
            int indexTo)
        {
            try
            {
                return this.WritePartData(fileName, indexFrom, indexTo);
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.WriteData(fileName, indexFrom, indexTo))\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool WritePartData(
            string fileName,
            int indexFrom,
            int indexTo)
        {
            BinaryWriter writer = null;
            try
            {
                this.fmtChunk_.BytesPerSec = (uint)(this.fmtChunk_.SamplesPerSec * this.fmtChunk_.BitsPerSample / WaveData.BYTE_SIZE * this.fmtChunk_.Channels);
                this.fmtChunk_.BlockAlign = (ushort)(this.fmtChunk_.BitsPerSample / WaveData.BYTE_SIZE * this.fmtChunk_.Channels);

                this.dataChunks_.SetChannelDataToChunkData(this.fmtChunk_, indexFrom, indexTo);

                byte[] uFileFormatID = Encoding.Unicode.GetBytes(WaveData.RIFF_ID);
                this.fileFormatID_ = Encoding.Convert(
                    Encoding.Unicode,
                    Encoding.ASCII,
                    uFileFormatID);

                this.fileSize_ = 
                    (uint)(
                    this.GetSize()
                    - sizeof(byte) * this.fileFormatID_.Length  //  fileFormatID
                    - sizeof(uint)                              //  fileSize
                    );

                byte[] uDataFormatID = Encoding.Unicode.GetBytes(WaveData.WAVE_ID);
                this.dataFormatID_ = Encoding.Convert(
                    Encoding.Unicode,
                    Encoding.ASCII,
                    uDataFormatID);

                writer = new BinaryWriter(File.Open(fileName, FileMode.Create, FileAccess.Write));
                writer.Write(this.fileFormatID_);
                writer.Write(this.fileSize_);
                writer.Write(this.dataFormatID_);

                if (!this.fmtChunk_.WriteByBinaryWriter(writer))
                {
                    throw new Exception(
                        "(WaveData.WritePartData)\n"
                        + "[FmtChunk.WriteByBinaryWriter(writer)]"
                        + "Write error occured.");
                }

                if (!this.dataChunks_.WriteByBinaryWriter(writer))
                {
                    throw new Exception(
                        "(WaveData.WritePartData)\n"
                        + "[DataChunks.WriteByBinaryWriter(writer)]"
                        + "Write error occured.");
                }

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveData.WritePartData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
            finally
            {
                if (writer != null)
                {
                    writer.Close();
                }
            }
        }

        #endregion  //  method
    }
}
