﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace NOPWORKS.Devel.Application.WaveCutter
{
    public class WaveDataChunk
    {
        //  chunk data
        private byte[] dataID_;             //  data ID[4byte]('d', 'a', 't', 'a')
        private uint dataSize_;             //  data size[4byte]
        private byte[] byteDatas_;          //  source data
        //  private variables
        private uint[][] channelDatas_;     //  waveform data of each channel(converted from source data)
        private long dataStartPosition_;    //  data start position in file
        private int bufferSize_;            //  data buffer size(byte)
        private int dataPointer_;           //  data buffer pointer

        #region constructor
        //
        //  constructor
        //

        public WaveDataChunk()
        {
            this.dataID_ = null;
            this.dataSize_ = 0;
            this.byteDatas_ = null;
            this.channelDatas_ = null;
            this.dataStartPosition_ = 0;
            this.bufferSize_ = 512 * 1024;  //  512kB
            this.dataPointer_ = 0;
            this.Clear();
        }

        #endregion  //  constructor

        #region property
        //
        //  property
        //

        public int BufferSize
        {
            get { return this.bufferSize_; }
            set { this.bufferSize_ = value; }
        }

        public byte[] ByteDatas
        {
            get { return this.byteDatas_; }
            set { this.byteDatas_ = value; }
        }

        public uint[][] ChannelDatas
        {
            get { return this.channelDatas_; }
            set { this.channelDatas_ = value; }
        }

        public int Channels
        {
            get
            {
                if (this.channelDatas_ == null)
                {
                    return 0;
                }
                else
                {
                    return this.channelDatas_.Length;
                }
            }
        }

        public byte[] DataID
        {
            get { return this.dataID_; }
            set { this.dataID_ = value; }
        }

        public string DataIDString
        {
            get { return Encoding.UTF8.GetString(this.dataID_); }
        }

        public int DataPointer
        {
            get { return this.dataPointer_; }
            set { this.dataPointer_ = value; }
        }

        public long DataStartPosition
        {
            get { return this.dataStartPosition_; }
            set { this.dataStartPosition_ = value; }
        }

        public uint DataSize
        {
            get { return this.dataSize_; }
            set { this.dataSize_ = value; }
        }

        #endregion  //  property

        #region method
        //
        //  method
        //

        public void Add(
            uint channel,
            uint value)
        {
            try
            {
                if (this.channelDatas_ == null)
                {
                    this.channelDatas_ = new uint[channel][];
                    this.channelDatas_[channel] = new uint[] { value };
                }
                else if (this.channelDatas_.Length < channel)
                {
                    throw new Exception(
                        "(WaveDataChunk.Add(channel, value))\n"
                        + "[ChannelDatas]: "
                        + "Too little number of array elements.");
                }
                else
                {
                    Array.Resize<uint>(
                        ref this.channelDatas_[channel], 
                        this.channelDatas_[channel].Length);
                    this.channelDatas_[channel][this.channelDatas_[channel].Length - 1] = value;
                }
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunk.Add)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message,
                    exp);
            }
        }

        public void Clear()
        {
            this.dataID_ = new byte[4];
            this.dataSize_ = 0;
            this.byteDatas_ = new byte[] { };
            this.bufferSize_ = WaveData.BYTEDATA_BUFFERSIZE;
            this.channelDatas_ = new uint[][] { };
        }

        public void ConvertToChannelData(
            WaveFmtChunk fmt)
        {
            try
            {
                int bytesPerSample = fmt.BytesPerSample;
                int dataSize = this.byteDatas_.Length / fmt.BytesPerSample / (int)fmt.Channels;
                this.channelDatas_ = new uint[fmt.Channels][];
                for (int i = 0; i < fmt.Channels; i++)
                {
                    this.channelDatas_[i] = new uint[dataSize];
                }

                int pos = 0;
                int channel = 0;
                int dstpos = 0;
                while (pos + bytesPerSample < this.byteDatas_.Length)
                {
                    uint n = WaveData.ConvertBytesToUInt(
                        this.byteDatas_, 
                        pos, 
                        pos + bytesPerSample - 1);

                    this.channelDatas_[channel][dstpos] = n;

                    if (channel == (fmt.Channels - 1))
                    {
                        ++dstpos;
                    }
                    channel = (++channel) % fmt.Channels;
                    pos += bytesPerSample;
                }
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunk.ConvertToChannelData)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public void CopyTo(
            WaveDataChunk dst)
        {
            dst.Clear();
            this.dataID_.CopyTo(dst.DataID, this.dataID_.Length);
            dst.DataSize = this.dataSize_;
            if (this.byteDatas_ != null)
            {
                dst.InitializeByteDatas(this.byteDatas_.Length);
                this.byteDatas_.CopyTo(dst.ByteDatas, this.byteDatas_.Length);
            }
            if (this.channelDatas_ != null)
            {
                uint channel = (uint)this.channelDatas_.Length;
                uint length = 0;
                for (uint i = 0; i < channel; i++)
                {
                    if (this.channelDatas_[i] != null)
                    {
                        if (length < this.channelDatas_[i].Length)
                        {
                            length = (uint)this.channelDatas_[i].Length;
                        }
                    }
                }
                dst.InitializeChannelDatas(channel, length);
                for (uint i = 0; i < channel; i++)
                {
                    Array.Copy(
                        this.channelDatas_[i], 
                        dst.ChannelDatas[i], 
                        this.channelDatas_[i].Length);
                }
            }
            dst.DataStartPosition = this.dataStartPosition_;
            dst.BufferSize = this.bufferSize_;
        }

        public uint GetSize()
        {
            return
                (uint)(
                sizeof(byte) * this.dataID_.Length          //  dataID_
                + sizeof(uint)                              //  dataSize_
                + sizeof(byte) * (this.byteDatas_.Length)   //  byteDatas_
                );
        }

        public void InitializeByteDatas(
            int length)
        {
            this.byteDatas_ = new byte[length];
        }

        public void InitializeChannelDatas(
            uint channel,
            uint length)
        {
            this.channelDatas_ = new uint[channel][];
            for (uint i = 0; i < channel; i++)
            {
                this.channelDatas_[i] = new uint[length];
            }
        }

        public long ReadByteDataFromBinaryReader(
            BinaryReader reader,
            long pos)
        {
            try
            {
                reader.BaseStream.Position = pos;
                this.byteDatas_ = reader.ReadBytes(this.bufferSize_);
                return reader.BaseStream.Position;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunk.ReadFromBinaryReader(reader, pos))\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        /// <summary>
        /// Reading sound source data from BinaryReader.
        /// </summary>
        /// <param name="reader">BinaryReader</param>
        /// <returns>True:read success</returns>
        public bool ReadFromBinaryReader(
            BinaryReader reader)
        {
            try
            {
                this.dataID_ = reader.ReadBytes(this.dataID_.Length);
                this.dataSize_ = reader.ReadUInt32();
                this.dataStartPosition_ = reader.BaseStream.Position;
                this.byteDatas_ = reader.ReadBytes((int)this.dataSize_);
                //reader.BaseStream.Position += this.dataSize_;

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunk.ReadFromBinaryReader(reader))\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool ReadHeaderFromBinaryReader(
            BinaryReader reader)
        {
            try
            {
                this.dataID_ = reader.ReadBytes(this.dataID_.Length);
                this.dataSize_ = reader.ReadUInt32();
                this.dataStartPosition_ = reader.BaseStream.Position;
                //this.byteDatas_ = reader.ReadBytes((int)this.dataSize_);
                reader.BaseStream.Position += this.dataSize_;

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunk.ReadHeaderFromBinaryReader(reader))\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool WriteByBinaryWriter(
            BinaryWriter writer)
        {
            try
            {
                this.dataSize_ = (uint)this.byteDatas_.Length;

                writer.Write(this.dataID_);
                writer.Write(this.dataSize_);
                writer.Write(this.byteDatas_);
                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveDataChunk.WriteByBinaryWriter(writer))\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        #endregion  //  method
    }
}
