﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace NOPWORKS.Devel.Application.WaveCutter
{
    public class WaveFmtChunk
    {
        public enum SoundDataFormat : ushort
        {
            unknown = 0,
            linear_PCM = 1
        }

        public enum SoundChannels : ushort
        {
            monaural = 1,
            stereo = 2
        }

        private byte[] chunkID_;            //  'f', 'm', 't', ' '
        private uint chunkSize_;            //  fmt chunk size(16(= 10 00 00 00):linear PCM)
        private ushort formatID_;           //  format ID(1(= 01 00):linear PCM)
        private ushort channels_;           //  channels of sound(1(= 01 00):monaural, 2(= 02 00):stereo)
        private uint samplesPerSec_;        //  sampling rate(i.e. 44.1k Hz = 44100(decimal number))
        private uint bytesPerSec_;          //  bytes per second(samplesPerSec_ * bitsPerSample_ / 8 * channels_)
        private ushort blockAlign_;         //  bytes per sample(bitsPerSample_ / 8 * channels_)
        private ushort bitsPerSample_;      //  bits per sample(little endian, i.e. 8bit=08 00)
        private ushort exSize_;             //  size of extra part(It doesn't exist for linear PCM)
        private byte[] exPart_;             //  extra part

        public WaveFmtChunk()
        {
            this.Clear();
        }

        #region property
        //
        //  property
        //

        public ushort BitsPerSample
        {
            get { return this.bitsPerSample_; }
            set { this.bitsPerSample_ = value; }
        }

        public ushort BlockAlign
        {
            get { return this.blockAlign_; }
            set { this.blockAlign_ = value; }
        }

        public int BytesPerSample
        {
            get { return ((int)this.bitsPerSample_) / WaveData.BYTE_SIZE; }
        }

        public uint BytesPerSec
        {
            get { return this.bytesPerSec_; }
            set { this.bytesPerSec_ = value; }
        }

        public ushort Channels
        {
            get { return this.channels_; }
            set { this.channels_ = value; }
        }

        public byte[] ChunkID
        {
            get { return this.chunkID_; }
            set { this.chunkID_ = value; }
        }

        public string ChunkIDString
        {
            get { return Encoding.UTF8.GetString(this.chunkID_); }
        }

        public uint ChunkSize
        {
            get { return this.chunkSize_; }
            set { this.chunkSize_ = value; }
        }

        public byte[] ExPart
        {
            get { return this.exPart_; }
            set { this.exPart_ = value; }
        }

        public ushort ExSize
        {
            get { return this.exSize_; }
            set { this.exSize_ = value; }
        }

        public ushort FormatID
        {
            get { return this.formatID_; }
            set { this.formatID_ = value; }
        }

        public uint SamplesPerSec
        {
            get { return this.samplesPerSec_; }
            set { this.samplesPerSec_ = value; }
        }

        #endregion  //  property

        #region method
        //
        //  method
        //

        public void Clear()
        {
            chunkID_ = new byte[4];
            chunkSize_ = 16;
            formatID_ = 1;
            channels_ = 1;
            samplesPerSec_ = 44100;
            bitsPerSample_ = 16;
            bytesPerSec_ = (ushort)(samplesPerSec_ * bitsPerSample_ / 8 * channels_);
            blockAlign_ = (ushort)(bitsPerSample_ / 8 * channels_);
            exSize_ = 0;
            exPart_ = null;
        }

        //public bool CheckHeader(
        //    BinaryReader reader)
        //{
        //    try
        //    {
        //        byte[] chunk_ID = reader.ReadBytes(WaveData.CHUNK_ID_DATASIZE);
        //        if (this.chunkID_.Length != chunk_ID.Length)
        //        {
        //            throw new Exception(
        //                "Wrong fmt chunk ID." 
        //                + "(" + Encoding.UTF8.GetString(chunk_ID) + ")");
        //        }
        //        else
        //        {
        //            for (int i = 0; i < this.chunkID_.Length; i++)
        //            {
        //                throw new Exception(
        //                    "Wrong fmt chunk ID."
        //                    + "[" + i.ToString() + "]"
        //                    + "(" + Encoding.UTF8.GetString(chunk_ID) + ")");
        //            }
        //        }

        //        uint chunk_size = reader.ReadUInt32();
        //        if (this.chunkSize_ != chunk_size)
        //        {
        //            throw new Exception(
        //                "Chunk size is different.\n\n"
        //                + "buffer value: " + this.chunkSize_.ToString()
        //                + "\n"
        //                + "file value: " + chunk_size.ToString());
        //        }

        //        ushort format_ID = reader.ReadUInt16();
        //        if (this.formatID_ != format_ID)
        //        {
        //            throw new Exception(
        //                "Format ID is different.\n\n"
        //                + "buffer value: " + this.formatID_.ToString()
        //                + "\n"
        //                + "file value: " + format_ID.ToString());
        //        }

        //        ushort channels = reader.ReadUInt16();
        //        if (this.channels_ != channels)
        //        {
        //            throw new Exception(
        //                "Channels is different.\n\n"
        //                + "buffer value: " + this.channels_.ToString()
        //                + "\n"
        //                + "file value: " + channels.ToString());
        //        }

        //        uint samples_per_sec = reader.ReadUInt32();
        //        if (this.samplesPerSec_ != samples_per_sec)
        //        {
        //            throw new Exception(
        //                "Samples/sec is different.\n\n"
        //                + "buffer value: " + this.samplesPerSec_.ToString()
        //                + "\n"
        //                + "file value: " + samples_per_sec.ToString());
        //        }

        //        uint bytes_per_sec = reader.ReadUInt32();
        //        ushort block_align = reader.ReadUInt16();
        //        ushort bits_per_sample = reader.ReadUInt16();
        //        if (format_ID != (ushort)SoundDataFormat.linear_PCM)
        //        {
        //            ushort ex_size = reader.ReadUInt16();
        //            if (ex_size > 0)
        //            {
        //                byte[] ex_part = reader.ReadBytes(ex_size);
        //            }
        //        }

        //        return true;
        //    }
        //    catch (Exception exp)
        //    {
        //        throw new Exception(
        //            "(WaveFmtChunk.ReadFromBinaryReader)\n"
        //            + "[" + exp.Source + "]"
        //            + exp.Message + "\n",
        //            exp);
        //    }
        //}

        public void CopyTo(
            WaveFmtChunk dst)
        {
            dst.Clear();
            for (int i = 0; i < this.chunkID_.Length; i++)
            {
                dst.ChunkID[i] = this.chunkID_[i];
            }
            dst.ChunkSize = this.chunkSize_;
            dst.FormatID = this.formatID_;
            dst.Channels = this.channels_;
            dst.SamplesPerSec = this.samplesPerSec_;
            dst.BitsPerSample = this.bitsPerSample_;
            dst.BytesPerSec = this.bytesPerSec_;
            dst.BlockAlign = this.blockAlign_;
            dst.ExSize = this.exSize_;
            dst.ExPart = new byte[this.exPart_.Length];
            for (int i = 0; i < this.exPart_.Length; i++)
            {
                dst.ExPart[i] = this.exPart_[i];
            }
        }

        public int GetMaxIntVolume()
        {
            return (int)(Math.Pow(2, this.bitsPerSample_ - 1) - 1);
        }

        public int GetMaxIntVolume2()
        {
            return (int)(Math.Pow(2, this.bitsPerSample_) - 1);
        }

        public uint GetSize()
        {
            return
                (uint)(
                sizeof(byte) * this.chunkID_.Length     //  chunkID_
                + sizeof(uint)                          //  chunkSize_
                + sizeof(ushort)                        //  formatID_
                + sizeof(ushort)                        //  channels_
                + sizeof(uint)                          //  samplesPerSec_
                + sizeof(uint)                          //  bytesPerSec_
                + sizeof(ushort)                        //  blockAlign_
                + sizeof(ushort)                        //  bitesPerSample_
                + ((this.exSize_ > 0)
                ? sizeof(ushort)                        //  exSize_
                + sizeof(byte) * this.exPart_.Length    //  exPart_
                : 0)
                );
        }

        public bool ReadFromBinaryReader(
            BinaryReader reader)
        {
            try
            {
                this.chunkID_ = reader.ReadBytes(this.chunkID_.Length);
                this.chunkSize_ = reader.ReadUInt32();

                long pos = reader.BaseStream.Position;

                this.formatID_ = reader.ReadUInt16();
                this.channels_ = reader.ReadUInt16();
                this.samplesPerSec_ = reader.ReadUInt32();
                this.bytesPerSec_ = reader.ReadUInt32();
                this.blockAlign_ = reader.ReadUInt16();
                this.bitsPerSample_ = reader.ReadUInt16();
                if (this.formatID_ != (ushort)SoundDataFormat.linear_PCM)
                {
                    this.exSize_ = reader.ReadUInt16();
                    if (this.exSize_ > 0)
                    {
                        this.exPart_ = new byte[this.exSize_];
                        this.exPart_ = reader.ReadBytes(this.exSize_);
                    }
                }

                reader.BaseStream.Position = pos + this.chunkSize_;

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveFmtChunk.ReadFromBinaryReader)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        public bool WriteByBinaryWriter(
            BinaryWriter writer)
        {
            try
            {
                this.bytesPerSec_ = (uint)(this.samplesPerSec_ * this.bitsPerSample_ / WaveData.BYTE_SIZE * this.channels_);
                this.blockAlign_ = (ushort)(this.bitsPerSample_ / WaveData.BYTE_SIZE * this.channels_);

                writer.Write(this.chunkID_);
                writer.Write(this.chunkSize_);
                writer.Write(this.formatID_);
                writer.Write(this.channels_);
                writer.Write(this.samplesPerSec_);
                writer.Write(this.bytesPerSec_);
                writer.Write(this.blockAlign_);
                writer.Write(this.bitsPerSample_);
                if (this.formatID_ != (ushort)SoundDataFormat.linear_PCM)
                {
                    if (this.exSize_ > 0)
                    {
                        writer.Write(this.exSize_);
                        writer.Write(this.exPart_);
                    }
                }

                return true;
            }
            catch (Exception exp)
            {
                throw new Exception(
                    "(WaveFmtChunk.WriteWithBinaryWriter)\n"
                    + "[" + exp.Source + "]"
                    + exp.Message + "\n",
                    exp);
            }
        }

        #endregion  //  method
    }
}
